
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

/** This class contains method main. An instance of this class is
    a window that contains the menu bar, message, and panel of
    images. An instance of Swing class JFrame is an independent 
    window on your monitor. */
public class ImageHandler extends JFrame {

	// the window title
	public static final String WINDOW_TITLE= "Image Handler";

	public static final String LOAD_INTERRUPTED= "Image loading interrupted";
	public static final String NORMAL= "Select or load an image";
	public static final String IMAGE_LIMIT_EXCEEDED= "Image limit exceeded";
	public static final String NO_IMAGE= "No image selected";
	
	JMenuBar menuBar= new JMenuBar();     // the menu bar
	JPanel panel;                         // the thumbnail panel
	JLabel message= new JLabel(NORMAL);   // the message label

	// The loaded images. A null array element indicates no image there
	ImageBox[] images= new ImageBox[10];

	// the array index of the current image (-1 if none)
	int curindex= -1;

	/** = this (provides a way to reference this inside inner classes */
	public ImageHandler getThis() {
		return this;
	}

    /** Constructor: The window, with a menu bar */
	public ImageHandler() {
		setTitle(WINDOW_TITLE);           // set window title
		panel= new JPanel(new GridLayout(2,5,10,10)); // set up thumbnail panel
		panel.setBorder(new EmptyBorder(0,10,0,10));
		
	    // set up the file menu
		    JMenu file= new JMenu("File");
			JMenuItem fileOpen= new JMenuItem("Open");
			fileOpen.addActionListener(new OpenActionListener());
			JMenuItem fileClose= new JMenuItem("Close");
            fileClose.addActionListener(new CloseActionListener());
			JMenuItem fileExit= new JMenuItem("Exit");
			fileExit.addActionListener(new ExitActionListener());
			file.add(fileOpen);
			file.add(fileClose);
			file.addSeparator();
			file.add(fileExit);
			menuBar.add(file);
			
	    // set up the image menu
			JMenu im= new JMenu("Image");
			JMenuItem inv= new JMenuItem("Invert");
			JMenuItem trans= new JMenuItem("Transpose");
			JMenuItem href= new JMenuItem("Horizontal Reflection");
			JMenuItem vref= new JMenuItem("Vertical Reflection");
			inv.addActionListener(new InvActionListener());
			trans.addActionListener(new TransActionListener());
			href.addActionListener(new HRefActionListener());
			vref.addActionListener(new VRefActionListener());
			im.add(inv);
			im.add(trans);
			im.add(href);
			im.add(vref);
			im.addSeparator();
			JMenuItem rest= new JMenuItem("Restore");
			rest.addActionListener(new RestActionListener());
			im.add(rest);
			menuBar.add(im);
			
	    // set up the help menu
			JMenu help= new JMenu("Help");
			JMenuItem showHelp= new JMenuItem("Show help");
			showHelp.addActionListener(new HelpActionListener());
			help.add(showHelp);
			menuBar.add(help);
			
	    // add components to frame
          panel.setPreferredSize(new Dimension(80,100));
	      
	      getContentPane().add(message,BorderLayout.NORTH);
		  getContentPane().add(panel,BorderLayout.SOUTH);
		  setJMenuBar(menuBar);
			
	    // add window closing event handler
		   addWindowListener(new WindowAdapter() {
               public void windowClosing(WindowEvent e) {
			   	      System.exit(0);}});
		pack();  // Tell the window to organize its contents
	    setVisible(true);  // Make the window visible
	  
		// REGISTER_LISTENERS
		   SymMouse aSymMouse= new SymMouse();
		   this.addMouseListener(aSymMouse);
	
		//{{REGISTER_LISTENERS
		   SymContainer aSymContainer= new SymContainer();
		   this.addContainerListener(aSymContainer);
		//}}
	}  

	// Add component c to gbl of component o (with constraints gbc)
	// at position (x,y). Component c takes w columns and r rows, 
	// and is weighted (wx, wy).
   private void add(Component c, Container o, GridBagLayout gbl,
			GridBagConstraints gbc,
			int x, int y, int w, int h, int wx, int wy) {
   	   gbc.gridx= x;
	   gbc.gridy= y;
	   gbc.gridwidth= w;
	   gbc.gridheight= h;
	   gbc.weightx= wx;
	   gbc.weighty= wy;
	   gbl.setConstraints(c, gbc);
	   o.add(c);
	}  

	/** Set the message field to m */
	public void setMessage(String m) {
	   message.setText(m);
       repaint();
	}

    /** display message m for 2 seconds */
	public void flashMessage(String m) {
		ErrorThread thread = new ErrorThread(m);
		thread.start();		  
	}

	/** the message flash must be run in a thread
	    so that the pause doesn't halt the GUI */
	class ErrorThread extends Thread {
   	
   	    String error;  // the message to be flashed

		// Constructor: A thread with error er
		public ErrorThread(String er) {
         error = er;
	    }
          
		// the run method
	    public void run() {
			setMessage(error);
		   try {
			   sleep(2000);
		   } catch (InterruptedException e) {}
		      setMessage(NORMAL);
			}
	    }

	/** Process menu File item Open */
   class OpenActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			int slot= findSlot();
			if (slot == -1) 
			   flashMessage(IMAGE_LIMIT_EXCEEDED);
			else {
				String im= ImageBox.loadImage();
				if (im != null) {
					images[slot]= new ImageBox(im,null,null,getThis(),slot);
					setCurrentImage(slot);
					displayThumbnails();  // refresh thumbnails
				}
			}
		}
	}

	/** Process menu File item Close */
	public class CloseActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			closeWindow();
		}
	}

	/** Process menu File item Exit */
   class ExitActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			System.exit(0);
		}
	}


	/** Process menu item Invert */
	class InvActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			if (curindex != -1) {
				images[curindex].invert();
			}
			else flashMessage(NO_IMAGE);
		}
	}	  

	// Process menu item Transpose
	class TransActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			if (curindex != -1) {
				images[curindex].transpose();
			}
			else flashMessage(NO_IMAGE);
		}
	}

	// Process menu item Horizontal reflection
	class HRefActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			if (curindex != -1) {
				images[curindex].hreflect();
			}
			else flashMessage(NO_IMAGE);
		}
	}

	// Process menu item Vertical reflection
	class VRefActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			if (curindex != -1) {
				images[curindex].vreflect();
			}
			else flashMessage(NO_IMAGE);
		}
	}

	/** Process menu item Restore */
	class RestActionListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			if (curindex != -1) 
				images[curindex].restoreImage();
			else flashMessage(NO_IMAGE);
		}
	}

    /** Process menu item Help */
	 class HelpActionListener implements ActionListener {
		 public void actionPerformed(ActionEvent e) {
		    String helps= "To load a new picture, use menu File item open. You can " +
		            "load .jpeg files and .gif files.]n" +
		        "\nTo close a picture, highlight it and use menu File item close.\n" +
		        "\nTo terminate the application, use either the close box or " +
		            "menu File item Exit.\n" +
		        "\nIn menu Image, you can use the following items, which operate " +
		            "on the currently highlighted picture." +
		        "\n  Invert, Transpose, Horizontal Reflection, and Vertical Reflection " +
		            "perform the obvious transformations on the picture." +
		        "\n  Restore restores the original picture.";
		        
			 TextWindow w= new TextWindow(helps,"Image Help", 60);
		 }
	 }	
	  
	/** Close the current Image window */
	public void closeWindow() {
		try {
		   images[curindex].dispose();
		   images[curindex]= null;
		   displayThumbnails();
		   curindex= -1;				 
		} catch (ArrayIndexOutOfBoundsException exc) {}
	}

    /** = the next available array slot to store an image.<br>
	    = -1 if no slot exists */
	public int findSlot() {
		// inv: no open slots in images[0..i-1]
		for (int i= 0; i < 10; i++) {
			if (images[i] == null)
				 return i;
		}
		return -1;
	}

	/** Make image n the current one */
	public void setCurrentImage(int n) {
		// hide previous image --if necessary
		   if (curindex != -1) {
				ImageBox temp= images[curindex].copy();
				images[curindex].dispose();				
				images[curindex]= temp;
				images[curindex].setVisible(false);
			}
		curindex= n;
		images[curindex].setVisible(true); // Make image visible
	}

	/** Display the thumbnails in the panel */
	public void displayThumbnails() {
		getContentPane().remove(panel);
		panel= new JPanel(new GridLayout(2,5,10,10));
		panel.setBorder(new EmptyBorder(0,10,0,10));
		
		for (int i=0; i<10; i++) {
			 if (images[i] != null)
				 panel.add(images[i].getThumbnail());
		}
		getContentPane().add(panel);
		pack();
		setVisible(true);
		repaint();
	}

	/** Method main simply creates an ImageHandler */
	public static void main(String args[]) {
		ImageHandler m= new ImageHandler();
	}

	class SymMouse extends java.awt.event.MouseAdapter {
		public void mouseClicked(java.awt.event.MouseEvent event) {
			Object object = event.getSource();
			if (object == ImageHandler.this)
				ImageHandler_mouseClicked(event);
		}
	}

	void ImageHandler_mouseClicked(java.awt.event.MouseEvent event) {
		// to do: code goes here.
	}

	class SymContainer extends java.awt.event.ContainerAdapter {
		public void componentAdded(java.awt.event.ContainerEvent event) {
			Object object = event.getSource();
			if (object == ImageHandler.this)
				ImageHandler_componentAdded(event);
		}
	}

	void ImageHandler_componentAdded(java.awt.event.ContainerEvent event) {
		// to do: code goes here.
	}
}

